/**********************************************************************
 * 
 * BootExtract.cpp
 *
 *
 * History:
 *  2005-08-04   v1    - First public release
 *  2006-01-05   v5    - FIxed a bug, so it is able to correctly read 1.2/1.44 and 2.88 MB emulations
 *
 **********************************************************************/
#include <windows.h>
#include <tchar.h>
#include <stdio.h>


#define CD_SEC_SIZE 0x800
#define CD_VIRT_SEC_SIZE 0x200
#define CD_BOOT_RECORD_VOL_SEC 0x11

#if _MSC_VER < 1400
#define _tcscpy_s(dest, bufLen, src) \
  do { \
    if (_tcslen(src) >= bufLen) \
      _tcsncpy(dest, src, bufLen); \
    else \
      _tcscpy(dest, src); \
    dest[bufLen-1] = 0; \
  } while(0)

#define _sntprintf_s _sntprintf
#define _tcscat_s(dst, len, src) _tcscat(dst, src)
#define _tprintf_s _tprintf

#endif


void WriteData(LPCTSTR szPrefix, LPCTSTR szAddName, LPVOID pBuffer, DWORD dwLen)
{
  TCHAR szFN[1024];
  _tcscpy_s(szFN, 1024, szPrefix);
  _tcscat_s(szFN, 1024, szAddName);

  _tprintf_s(_T("Writing '%s'\n"), szFN);

  HANDLE hWrite = CreateFile(szFN, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, 0, NULL);
  if (hWrite != INVALID_HANDLE_VALUE)
  {
    DWORD dwWritten;
    if (WriteFile(hWrite, pBuffer, dwLen, &dwWritten, NULL) == FALSE)
    {
      _tprintf_s(_T("Could not write the file '%s', Error: %d\n"), szFN, GetLastError());
    }
    CloseHandle(hWrite);
  }
  else
    _tprintf_s(_T("Could not write the file '%s', Error: %d\n"), szFN, GetLastError());
}

DWORD Read(HANDLE hCD, DWORD startPos, DWORD bytesToRead, LPVOID pBuffer)
{
  DWORD dwRet = SetFilePointer(hCD, startPos, NULL, FILE_BEGIN);
  if (dwRet != startPos)
    return ERROR_SEEK;

  DWORD dwRead;
  BOOL bRet = ReadFile(hCD, pBuffer, bytesToRead, &dwRead, NULL);
  if (bRet == FALSE)
    return GetLastError();
  if (bytesToRead != dwRead)
    return ERROR_NO_DATA;
  return ERROR_SUCCESS;
}


void ProcessEntry(LPCTSTR szPrefix, HANDLE hCD, unsigned char *buffer)
{
  if (buffer[0] == 0x88)
    printf("  Boot-Indicator: Bootable\n");
  else
  {
    printf("  Boot-Indicator: Not Bootable\n");
    return;
  }

  DWORD dwStartingSector = *((DWORD*)&buffer[8]);
  WORD wVirtSectorCount = *((WORD*)&buffer[6]);
  WORD wSegment = *((WORD*)&buffer[2]);

  WORD wReadSecCount = wVirtSectorCount;


  _tprintf_s(_T("  Emulation: "));
  TCHAR szEmuMode[100];
  switch(buffer[1] & 0x0F)
  {
  case 0: 
    _tprintf_s(_T("No emulation\n")); 
    _tcscpy_s(szEmuMode, 100, _T("NoEmulation")); 
    break;
  case 1:
    _tprintf_s(_T("1.2 meg diskette\n")); 
    _tcscpy_s(szEmuMode, 100, _T("1.2MegDisk")); 
    wReadSecCount = 0x50 * 0x02 * 0x0F;
    break;
  case 2: 
    _tprintf_s(_T("1.44 meg diskette\n")); 
    _tcscpy_s(szEmuMode, 100, _T("1.44MegDisk")); 
    wReadSecCount = 0x50 * 0x02 * 0x12;
    break;
  case 3: 
    _tprintf_s(_T("2.88 meg diskette\n")); 
    _tcscpy_s(szEmuMode, 100, _T("2.88MegDisk")); 
    wReadSecCount = 0x50 * 0x02 * 0x24;
    break;
  case 4: 
    _tprintf_s(_T("Hard disk (drive 80)\n")); 
    _tcscpy_s(szEmuMode, 100, _T("HardDisk")); 
    break;
  default: 
    _tprintf_s(_T("Unknown\n")); 
    _tcscpy_s(szEmuMode, 100, _T("EmuUnknown")); 
    break;
  }

  printf("  Load segment: 0x%4.4X\n", wSegment );
  printf("  Sector count: %d (%d bytes)\n", wVirtSectorCount, wVirtSectorCount*CD_VIRT_SEC_SIZE);
  printf("  Starting sector: 0x%x\n", dwStartingSector);
  if (wReadSecCount != wVirtSectorCount)
  {
    printf("  Sector count to read: %d (%d bytes)\n", wReadSecCount, wReadSecCount*CD_VIRT_SEC_SIZE);
  }

  unsigned char *pBootSec;
  DWORD dwLen = wReadSecCount * CD_VIRT_SEC_SIZE;
  pBootSec = new unsigned char[dwLen];


  // read the "boot sector"
  if (Read(hCD, CD_SEC_SIZE * dwStartingSector, dwLen, pBootSec) != ERROR_SUCCESS)
  {
    delete [] pBootSec;
    return;
  }

  TCHAR szAdd[1024];
  _sntprintf_s(szAdd, 1024, _T("%s__Segment-%4.4X__SecCount-%d.bin"), szEmuMode, wSegment, wVirtSectorCount);

  WriteData(szPrefix, szAdd, pBootSec, dwLen);
  delete [] pBootSec;
}

void Header()
{
  _tprintf_s(_T("CD/DVD-BootImage-Extractor v1.0 (c) Jochen Kalmbach\n"));
  _tprintf_s(_T("http://blog.kalmbachnet.de/\n"));
  _tprintf_s(_T("\n"));
}


void Usage()
{
  TCHAR szExeName[1024];
  TCHAR szFName[_MAX_FNAME];
  if (GetModuleFileName(NULL, szExeName, 1024) != 0)
  {
		_tsplitpath(szExeName, NULL, NULL, szFName, NULL);
  }
  else
  {
    _tcscpy_s(szFName, _MAX_FNAME, _T("BootExtract"));
  }

  _tprintf_s(_T("Usage:\n"));
  _tprintf_s(_T("  %s [switches] <source>\n"), szFName);
  _tprintf_s(_T("\n"));
  _tprintf_s(_T("  source: The source file (*.iso) or CD/DVD-drive to extract images from\n"));
  _tprintf_s(_T("\n"));
  _tprintf_s(_T("  switches: -b   Also extract bootrecord and bootcatalog\n"));
  _tprintf_s(_T("\n"));
  _tprintf_s(_T("Examples:\n"));
  _tprintf_s(_T("  %s z:\n"), szFName);
  _tprintf_s(_T("  %s redhat71.iso\n"), szFName);
}

int _tmain(int argc, TCHAR *argv[])
{
  Header();

  if (argc < 2)
  {
    Usage();
    return 1;
  }

  bool bExtractAll = false;
  for(int i=1; i<argc-1; i++)
  {
    if (_tcsicmp(argv[i], _T("-b")) == 0)
      bExtractAll = true;
  }
  LPCTSTR szFN = argv[argc-1];

  TCHAR szFileName[1024];
  TCHAR szOutFileName[1024];
  if ( (_tcslen(szFN) == 2) && (szFN[1] == ':') )
  {
    _sntprintf_s(szFileName, 1024, _T("\\\\.\\%s"), szFN);
    _sntprintf_s(szOutFileName, 1024, _T("Drive-%c."), szFN[0]);
  }
  else
  {
    _tcscpy_s(szFileName, 1024, szFN);
    _tcscpy_s(szOutFileName, 1024, szFN);
    _tcscat_s(szOutFileName, 1024, _T("."));
  }

  // open the CD-ROM drive
  HANDLE hCD = CreateFile(szFileName, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, NULL);
  if (hCD == INVALID_HANDLE_VALUE)
  {
    _tprintf_s(_T("Could not open the file/drive (%s), Error: %d\n"), szFileName, GetLastError());
    return 1;
  }

  unsigned char buffer[CD_SEC_SIZE];
  DWORD dwRet;

  _tprintf_s(_T("Reading boot-record-volume from sector 0x%x\n"), CD_BOOT_RECORD_VOL_SEC);
  dwRet = Read(hCD, CD_SEC_SIZE * CD_BOOT_RECORD_VOL_SEC, CD_SEC_SIZE, buffer);
  if (dwRet != ERROR_SUCCESS)
  {
    _tprintf_s(_T("Could not read the file/drive (%s), Error: %d\n"), szFileName, dwRet);
    CloseHandle(hCD);
    return 1;
  }

  // read the "boot record volume"
  // and check if this is a valid boot-record volume description:
  if ( 
    (buffer[0] != 0) || 
    (buffer[6] != 1)
    )
  {
    _tprintf_s(_T("Invalid boot-record-volume!\n"));
    CloseHandle(hCD);
    return 1;
  }
  buffer[6] = 0;
  buffer[30] = 0;
  if ( 
    (strcmp((char*)&buffer[1], "CD001") != 0) || 
    (strcmp((char*)&buffer[7], "EL TORITO SPECIFICATION") != 0) ) 
  {
    _tprintf_s(_T("Invalid signature in boot-record-volume!\n"));
    CloseHandle(hCD);
    return 1;
  }

  _tprintf_s(_T("  Valid boot-record-volume found...\n"));
  if (bExtractAll == true)
    WriteData(szOutFileName, _T("boot-record-volume.bin"), buffer, CD_SEC_SIZE);

  // ok, it is valid... now get the sector of the boot-catalog
  DWORD dwBootCatalogSector = *(DWORD*) (&buffer[0x47]);

  // Read the boot-catalog:
  _tprintf_s(_T("\nReading boot-catalog from sector 0x%x\n"), dwBootCatalogSector);
  dwRet = Read(hCD, CD_SEC_SIZE * dwBootCatalogSector, CD_SEC_SIZE, buffer);
  if (dwRet != ERROR_SUCCESS)
  {
    _tprintf_s(_T("Could not read the file/drive (%s), Error: %d\n"), szFileName, dwRet);
    CloseHandle(hCD);
  }

  // Display the "Validation-Entry":
  if ( 
    (buffer[0] != 1) || 
    (buffer[0x1E] != 0x55) ||
    (buffer[0x1F] != 0xAA) )
  {
    _tprintf_s(_T("Invalid validation-entry in boot-catalog!\n"));
    CloseHandle(hCD);
    return 1;
 }
  switch(buffer[1])
  {
  case 0:
    _tprintf_s(_T("  PlatformID: 80x86\n"));
    break;
  case 1:
    _tprintf_s(_T("  PlatformID: PowerPC\n"));
    break;
  case 2:
    _tprintf_s(_T("  PlatformID: Mac\n"));
    break;
  default:
    _tprintf_s(_T("  PlatformID: Unknown\n"));
    break;
  }
  buffer[0x1C] = 0;
#ifdef _UNICODE
  _tprintf_s(_T("  ID-String: %S\n"), &buffer[4]);
#else
  _tprintf_s(_T("  ID-String: %s\n"), &buffer[4]);
#endif

  if (bExtractAll == true)
    WriteData(szOutFileName, _T("boot-catalog.bin"), buffer, CD_SEC_SIZE);

  // Display the Initial/Default-Entry:
  _tprintf_s(_T("\nProcessing Initial/Default-Entry...\n"));
  ProcessEntry(szOutFileName, hCD, &buffer[0x20]);

  CloseHandle(hCD);
} 
